---
title: Get JSON response for mobile and other non-browser traffic · Cloudflare
  Waiting Room docs
description: "If you need to manage traffic in a non-browser environment such as
  a mobile app or web app, Cloudflare provides a JSON-friendly waiting room that
  can be consumed via your API endpoints:"
lastUpdated: 2025-08-15T20:11:52.000Z
chatbotDeprioritize: false
tags: JSON
source_url:
  html: https://developers.cloudflare.com/waiting-room/how-to/json-response/
  md: https://developers.cloudflare.com/waiting-room/how-to/json-response/index.md
---

If you need to manage traffic in a non-browser environment such as a mobile app or web app, Cloudflare provides a JSON-friendly waiting room that can be consumed via your API endpoints:

1. When a user is queued, we return our own JSON response.
2. When a user leaves the waiting room, we forward the request to your origin server and return the response from your origin server (be it JSON, XML, an HTML page, etc.).

In order to consume the waiting room response in the JSON format, the following steps must be taken:

## Step 1 – Enable JSON response

To receive a JSON response, you first need to enable that option in your waiting room.

* **Via the dashboard**: When [customizing a waiting room](https://developers.cloudflare.com/waiting-room/how-to/customize-waiting-room/), enable **JSON Response**.
* **Via the API**: When [creating a waiting room](https://developers.cloudflare.com/api/resources/waiting_rooms/methods/create/), set `json_response_enabled` to true.

## Step 2 – Get JSON data

Make a request to your waiting room endpoint with the header `Accept: application/json`. Note that the header has to match exactly `Accept: application/json`. If it is anything else or has any additional content such as `Accept: application/json, text/html` the response will not return in the JSON format. You must retry the request every `refreshIntervalSeconds` in order for users to advance in the queue.

```bash
curl "https://example.com/waitingroom" \
--header "Accept: application/json"
```

```json
{
  "cfWaitingRoom": {
    "inWaitingRoom": true,
    "waitTime": 5,
    "waitTimeKnown": true,
    "waitTimeFormatted": "5 minutes",
    "queueIsFull": false,
    "queueAll": false,
    "lastUpdated": "2021-08-03T23:46:00.000Z",
    "refreshIntervalSeconds": 20
  }
}
```

## Cookies in the request header

Waiting Room is driven by a waiting room cookie that determines the position of the user in the queue. Because of this, the cookie is updated in the response headers for each request. For each request to an endpoint protected by Waiting Room, the application must include the up-to-date cookie retrieved during the previous request. This is mandatory regardless of a user having been queued or not. If a request does not include a cookie, the waiting room will assume this is a new user and will return a new cookie in the response header. Consequently, this will place the user at the end of the queue. Thus, when consuming the waiting room in a non-browser environment it is important to include the waiting room cookie in the request header and keep it updated after each request.

Refer to the [Waiting Room cookies](https://developers.cloudflare.com/waiting-room/reference/waiting-room-cookie/), for more information.

## Advancing in the queue

In a browser environment, the page automatically refreshes every `refreshIntervalSeconds` to ensure that the user advances in the queue. In a non-browser environment, where the Waiting Room JSON-friendly API is being consumed, it is expected that your backend service (or API) also refreshes/makes a request to the Waiting Room configured endpoint every `refreshIntervalSeconds` to ensure the advancing of the user in the queue.

These are some of the places where the JSON-friendly response can be consumed (this list is not exhaustive):

1. In a mobile app traffic

   * **Integrate Waiting Room variables** – Create a new template in your mobile app to receive the JSON response. For a full list of these variables, refer to the `json_response_enabled` parameter in the [Cloudflare API docs](https://developers.cloudflare.com/api/resources/waiting_rooms/methods/create/).
   * **Allow cookies** – As mentioned above, a waiting room [requires cookies](https://developers.cloudflare.com/waiting-room/reference/waiting-room-cookie/), and your mobile app will need to support cookies. For ease of use, consider using a cookie manager like [CookieJar](https://pkg.go.dev/net/http#CookieJar).
   * **Consume JSON data** - Make a request to the Waiting Room endpoint with the `Accept: application/json` header.

2. Inside Cloudflare Workers (or in your own backend service)

   * **Integrate Waiting Room variables** – Expect a JSON response in your backend API. For a full list of these variables, refer to the `json_response_enabled` parameter in the [Cloudflare API docs](https://developers.cloudflare.com/api/resources/waiting_rooms/methods/create/).

   * **Include cookies in the request header** – As mentioned above, a waiting room [requires cookies](https://developers.cloudflare.com/waiting-room/reference/waiting-room-cookie/), and your backend API will need to support cookies. For ease of use, consider using a cookie manager like [CookieJar](https://pkg.go.dev/net/http#CookieJar).

   * **Enable JSON response** - Via the dashboard or via the API.

   * **Consume JSON data** - Make a request to the Waiting Room endpoint with the `Accept: application/json` header.

     Here is an example, demonstrating the usage of the waiting room endpoint inside a Worker. The request headers include the necessary `accept` and `cookie` header values that are required by the Waiting Room API. The accept header ensures that a JSON-friendly response is returned, if a user is queued. Otherwise, if the request is sent to the origin, then whatever the response origin returns gets returned back. In this example, a hardcoded `__cfwaitingroom` value is embedded in the cookie field. In a real-life application, however, we expect that a cookie returned by the Waiting Room API is used in each of the subsequent requests to ensure that the user is placed accordingly in the queue and let through to the origin when it is the users turn.

```javascript
const waitingroomSite = "https://examples.cloudflareworkers.com/waiting-room";


export default {
  async fetch(request, env, ctx) {
    const init = {
      headers: {
        accept: "application/json",
        cookie: "__cfwaitingroom=F)J@NcRfUjXnZr4u7x!A%D*G-KaPdSgV",
      },
    };


    return fetch(waitingroomSite, init)
      .then((response) => response.json())
      .then((response) => {
        if (response.cfWaitingRoom.inWaitingRoom) {
          return Response("in waiting room", { "content-type": "text/html" });
        }
        return new Response(response);
      });
  },
};
```

Note

Only Advanced Waiting Room customers can support JSON-friendly format with their waiting rooms. For more details, refer to our [Plans page](https://developers.cloudflare.com/waiting-room/plans/).
